/*
 * Conversation.java
 *
 * Created on February 14, 2003, 10:58 AM
 */

package za.org.dragon.exodus;

/** Conversation is a class designed to represent the possible steps in a HTTP or
 * HTTPS conversation via a proxy.
 *
 * It can contain a client request, received from the browser, potentially a server
 * request, if the request was modified by the proxy in any way.
 * It can contain a server response, received from the server as a direct result of
 * either the serverRequest or the clientRequest being sent to the server.
 * It can also contain a description of the conversation.
 * @author rdawes
 */
public class Conversation {
        
    private Request clientRequest = null; // The entire request received from the client 
    
    private Request serverRequest = null; // The entire request sent to the server

    private Response serverResponse = null; // the entire response received from the server
    
    private Response clientResponse = null; // the entire response sent to the client
    
    private String description; // A description of this conversation
    
    private String origin = null; // Where this conversation originates
       
    private String id;
    
    private boolean isAnalysed = false;
    
    /** Creates a new instance of Conversation */
    public Conversation() {
    }
    
    /** If the proxy received a request from the browser via the network, that request
     * will be stored as the clientRequest.
     * @return returns the client request, if any. There may be no client request if the
     * conversation was generated by the proxy itself. In that case, there must be a
     * serverRequest
     */    
    public Request getClientRequest() {
        return clientRequest;
    }
    
    /** Sets the clientRequest part of the conversation
     * @param request a Request object representing the client request
     */    
    public void setClientRequest(Request request) {
        clientRequest = request;
    }

    /** If the proxy modifies the request in any way, it should be recorded as the
     * serverRequest
     * @return the modified request, if any, otherwise returns null
     */    
    public Request getServerRequest() {
        return serverRequest;
    }
    
    /** If the proxy modifies the request, it is stored as the "serverRequest"
     * @param request a Request object representing the now-modified client request
     */    
    public void setServerRequest(Request request) {
        serverRequest = request;
    }

    /** Once the request has been sent to the server, the response received will be
     * stored as the serverResponse
     * @return a Response object representing the response received from the server
     */    
    public Response getServerResponse() {
        return serverResponse;
    }
    
    /** Once the request has been sent to the server, the response received will be
     * stored as the serverResponse
     * @param response a Response object representing the response received from the server
     */    
    public void setServerResponse(Response response) {
        serverResponse = response;
    }

    /** If the proxy modifies the response received from the server, it should be
     * recorded as the clientResponse.
     * @return a Response object representing the now-modified server Response
     */    
    public Response getClientResponse() {
        return clientResponse;
    }
    
    /** If the proxy modifies the response received from the server, it should be
     * recorded as the clientResponse.
     * @param response a Response object representing the now-modified server Response
     */    
    public void setClientResponse(Response response) {
        clientResponse = response;
    }

    /** We can also store an optional comment describing this conversation, perhaps a
     * reason why this conversation happened.
     * @return a String containing the description, if any, otherwise null
     */    
    public String getDescription() {
        if (description != null) {
            return description;
        } else { 
            return "";
        }
    }
    
    /** We can also store an optional comment describing this conversation, perhaps a
     * reason why this conversation happened.
     * @param description a String containing the description for this conversation
     */    
    public void setDescription(String description) {
        this.description = description;
    }
    
    public String getOrigin() {
        return origin;
    }
    
    public void setOrigin(String origin) {
        this.origin = origin;
    }
    
    /** Each conversation should (must) have a unique id. Normally it is generated
     * internally.
     * @return An "opaque" conversation id. It can only really be used for display, or maybe
     * for sorting on
     */    
    public String getID() {
        return this.id;
    }

    public void setID(String id) {
        this.id = id;
    }
    
    /** used by the model to determine whether to analyse this conversation for links,
     * forms, scripts, etc.
     * This should not really be here, it should be in the model somewhere. :-(
     * @return Whether we have looked for interesting things within this conversation, or not.
     */    
    public boolean isAnalysed() {
        return isAnalysed;
    }
}
