/*
 * $Id: Exodus.java,v 1.10 2003/06/11 05:16:40 rdawes Exp $
 * Exodus.java
 *
 * Created on February 16, 2003, 8:18 AM
 */

package za.org.dragon.exodus;

import za.org.dragon.exodus.Model;
import za.org.dragon.exodus.ExodusUI;
import za.org.dragon.exodus.Proxy;

import java.io.IOException;
import java.io.File;
import java.io.InputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileNotFoundException;
import java.util.Properties;
import java.util.logging.Logger;

import java.lang.ClassLoader;

/**
 *
 * @author  rdawes
 */
public class Exodus {
    
    private Model model;
    private BackingStore bs;
    private ExodusPlugin[] plugins;
    private Properties properties;
    private Logger logger = Logger.getLogger("za.org.exodus.Exodus");
    private Proxy proxy;
    
    Exodus()  {
        
        model = new Model();
        proxy = new Proxy(this);
        
        properties = readProperties();
        configure(properties);

        plugins = loadPlugins(model, properties);
        proxy.setPlugins(plugins);
        
        // Spawn a thread to run the proxy
        new Thread(proxy).start();
    }
    
    public Model getModel() {
        return model;
    }
    
    private ExodusPlugin loadPlugin(String className) {
        ExodusPlugin plugin = null;
        try {
            Class classDefinition = Class.forName(className);
            plugin = (ExodusPlugin) classDefinition.newInstance();
        } catch (InstantiationException e) {
            logger.severe(e.toString());
        } catch (IllegalAccessException e) {
            logger.severe(e.toString());
        } catch (ClassNotFoundException e) {
            logger.severe(e.toString());
	}
	return plugin;
    }
    
    private ExodusPlugin[] loadPlugins(Model model, Properties props) {
        int pluginCount = 0;
        try {
            pluginCount = Integer.parseInt(props.getProperty("Exodus.plugins"));
        } catch (NumberFormatException nfe) {
            logger.severe("Error determining the number of plugins to load! " + nfe);
            System.exit(1);
        }
        
        ExodusPlugin[] plugins = new ExodusPlugin[pluginCount+1];
        for (int i = 0; i < pluginCount; i++) {
            String pluginClass = props.getProperty("Exodus.plugin." + i);
            if (pluginClass != null) {
                plugins[i] = loadPlugin(pluginClass);
            } else {
                plugins[i] = null;
            }
            if (plugins[i] == null) {
                logger.severe("Failed to load Exodus.plugin." + i + " : " + pluginClass);
                System.exit(1);
            } else {
                logger.info("Loaded Exodus.plugin." + i + " : " + pluginClass);
                plugins[i].init(this);
                plugins[i].setProperties(props);
            }
        }
        return plugins;
    }
    
    public ExodusPlugin[] getPlugins() {
        return plugins;
    }
    
    public Properties readProperties() {
        // Properties work as follows :
        //    Read the defaults from the .jar. This MUST exist.
        //    Then look for a props file in the user's home directory, and load it if it exists
        
        String sep = System.getProperty("file.separator");
        String props = "Exodus.properties";
        String home = System.getProperty("user.home");

        InputStream is = null;
        
        Properties defaults = new Properties();
        is = ClassLoader.getSystemResourceAsStream(props);
        if (is == null) {
            logger.severe("Couldn't find the default props file!");
            System.exit(1);
        }
        try {
            defaults.load(is);
        } catch (IOException ioe) {
            logger.severe("Error reading default properties file " + ioe);
            System.exit(1);
        }
        
        Properties homeProperties = new Properties(defaults);
        try {
            is = new FileInputStream(home + sep + props);
        } catch (FileNotFoundException fnfe) {
            is = null;
        }
        if (is != null) {
            try {
                homeProperties.load(is);
            } catch (IOException ioe) {
                logger.severe("IOError reading " + home + sep + props + " : " + ioe);
            }
        }

        return homeProperties;
    }
    
    public void setProperties(Properties properties) {
        this.properties = properties;
    }
    
    public Properties getProperties() {
        if (properties == null) {
            logger.severe("properties == null");
            System.exit(1);
        }
        return properties;
    }
    
    public void writeProperties() {
        String sep = System.getProperty("file.separator");
        String props = "Exodus.properties";
        String home = System.getProperty("user.home");
        FileOutputStream fos;
        try {
            fos = new FileOutputStream(home + sep + props);
        } catch (Exception e) {
            fos = null;
        }
        if (fos == null) {
            logger.severe("Couldn't write to " + home + sep + props);
        } else {
            try {
                properties.store(fos,"Properties for Exodus Proxy");
                fos.close();
            } catch (IOException ioe) {
                logger.severe("IOError reading " + home + sep + props + " : " + ioe);
            }
        }
    }
    
    public void configure(Properties props) {
        String prop = "Exodus.listenHost";
        String listenHost = props.getProperty(prop);
        try {
            prop = "Exodus.listenPort";
            int listenPort = Integer.parseInt(props.getProperty(prop));
            this.proxy.setListenPort(listenHost,listenPort);
        } catch (NumberFormatException nfe) {
            System.out.println("Error parsing " + prop + " from properties");
        }
        
        prop = "Exodus.httpProxy";
        String[] proxy = props.getProperty(prop).split(":");
        if (proxy.length == 2) {
            try {
                URLFetcher.setHttpProxy(proxy[0], Integer.parseInt(proxy[1]));
            } catch (NumberFormatException nfe) {
                System.out.println("Error parsing " + prop + " from properties");
            }
        } else {
			URLFetcher.setHttpProxy(null,0);
		}
        prop = "Exodus.httpsProxy";
        proxy = props.getProperty(prop).split(":");
        if (proxy.length == 2) {
            try {
                URLFetcher.setHttpsProxy(proxy[0], Integer.parseInt(proxy[1]));
            } catch (NumberFormatException nfe) {
                System.out.println("Error parsing " + prop + " from properties");
            }
        } else {
			URLFetcher.setHttpsProxy(null,0);
        }
        prop = "Exodus.noProxy";
        URLFetcher.setNoProxy(props.getProperty(prop));
    }
        
    public void load(String session) {
        BackingStore bs = new BackingStore(session);
        model.clearSession();
        for (int i=0; i< plugins.length; i++) {
            if (plugins[i] != null) {
                plugins[i].clearSession();
            }
        }
        try {
            String list[] = bs.listConversations();
            if (list != null && list.length>0) {
                for (int i=0; i<list.length; i++) {
                    model.setConversationID(list[i]);
                    // System.out.println("Reading conversation " + list[i]);
                    Conversation c = bs.readConversation(list[i]);
                    Request req = c.getClientRequest();
                    if (req != null) model.setClientRequest(list[i],req);
                    req = c.getServerRequest();
                    if (req != null) model.setServerRequest(list[i],req);
                    Response resp = c.getServerResponse();
                    if (resp != null) model.setServerResponse(list[i],resp);
                    resp = c.getClientResponse();
                    if (resp != null) model.setClientResponse(list[i],resp);
                    String description = c.getDescription();
                    model.setDescription(list[i], description);
                    String origin = c.getOrigin();
                    model.setOrigin(list[i], origin);
                    Thread.currentThread().yield();
                }
            } else {
                logger.info("No conversations here. Initing the directory");
                bs.init();
            }
            model.setBackingStore(bs);
        } catch (IOException ioe) {
            System.err.println("Error reading conversations!");
            System.exit(1);
        }
    }
    
    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        Exodus exodus = new Exodus();
        
        // Create the UI, and spawn an independent thread for it to operate in
        ExodusUI exodusui = new ExodusUI(exodus);
        new Thread(exodusui).start();
        
        // If we were given a parameter on the command line, treat it as a session to load
        if (args.length == 1) {
            exodus.load(args[0]);
        }
    }
}
