package za.org.dragon.exodus;

import za.org.dragon.exodus.Conversation;
import za.org.dragon.exodus.URLInfo;
import za.org.dragon.exodus.BackingStore;

//import java.util.Hashtable;
import java.util.Observable;
import java.util.ArrayList;
import java.util.Vector;

import java.util.TreeMap;
import java.util.Map;
import java.util.Collections;
import java.util.Iterator;
import java.net.URL;
import java.net.MalformedURLException;

import java.util.Timer;
import java.util.TimerTask;

import java.io.IOException;

import javax.swing.DefaultListModel;
import javax.swing.table.TableModel;
import javax.swing.table.AbstractTableModel;

import java.util.logging.Logger;

import javax.swing.tree.TreePath;
import javax.swing.tree.TreeModel;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.DefaultMutableTreeNode;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;



// Model.java

/** Model represents the most significant part of the Exodus system. It contains the
 * conversations, as well as the information about all the URLs that have been
 * seen. It is created as an Observable object, so that plugins can be notified of
 * any new Conversations, or URLInfos.
 */
public class Model extends Observable {
    
    private Map conversation;         // records all the conversations seen.
    private ArrayList conversationList;  // maintains a FIFO list of cached conversations
    private Vector _conversationData = new Vector(1);  // Keeps the conversation data, is referenced by the ConversationTableModel
    private Map urlinfo;              // maps urls to attrs
    private ArrayList changed = new ArrayList(1);
    private BackingStore backingStore = null;
    private int _cachesize = 10;
    
    private Logger logger = Logger.getLogger("za.org.dragon.exodus.Model");
    
    // These do nothing yet, but they should replace the urlinfo and conversation maps above
    // Interested plugins will get these models, and simply listen to whatever model events
    // they care about
    private ConversationTableModel _ctm = new ConversationTableModel();
    private WebTreeModel _wtm = new WebTreeModel();
    
    /**
     *  Constructor
     */
    public Model( ) {
        conversation = Collections.synchronizedMap(new TreeMap());
        conversationList = new ArrayList(1);
        urlinfo = Collections.synchronizedMap(new TreeMap());
    } // constructor Model
    
    /** Initialises the lists of conversations, URLInfos, etc */
    public void clearSession() {
        // clear the conversation cache
        conversation.clear();
        conversationList.clear();
        // clear the conversationtablemodel
        _conversationData.clear();
        _ctm.fireTableDataChanged();
        // clear the URLInfo. Will go away when the site tree model is finished?
        urlinfo.clear();
        _wtm.clear();
    }
    
    private String currentConversationID = "00000";
    
    private synchronized String getNextConversationID() {
        String id;
        // increment the current id
        currentConversationID = String.valueOf(Integer.parseInt(currentConversationID)+1);
        for (int i=currentConversationID.length(); i<5; i++) {
            currentConversationID = "0" + currentConversationID;
        }
        return currentConversationID;
    }
    
    public String newConversationID() {
        String id = getNextConversationID();
        if (backingStore != null) {
            try {
                backingStore.writeConversation(null,  id);
            } catch (IOException ioe) {
                logger.warning("Couldn't create the conversation directory for " + id + " : " + ioe);
            }
        }
        return id;
    }
    
    public void setConversationID(String id) {
        if (Integer.parseInt(id) > Integer.parseInt(currentConversationID)) {
            currentConversationID = id;
        }
    }
    
    public void setClientRequest(String conversationID, Request request) {
        Conversation c = (Conversation) conversation.get(conversationID);
        if (c == null) {
            c = new Conversation();
            c.setID(conversationID);
        }
        c.setClientRequest(request);
        cacheConversation(conversationID,c);
        if (backingStore != null) {
            try {
                backingStore.writeRequest(request, conversationID, "fromclient");
            } catch (IOException ioe) {
                logger.warning("Error writing client request for " + conversationID + " : " + ioe);
            }
        }
    }
    
    public Request getClientRequest(String conversationID) {
        Conversation c = getConversation(conversationID);
        if (c != null) {
            return c.getClientRequest();
        }
        return null;
    }
    
    public void setServerRequest(String conversationID, Request request) {
        Conversation c = (Conversation) conversation.get(conversationID);
        if (c == null) {
            c = new Conversation();
            c.setID(conversationID);
        }
        c.setServerRequest(request);
        cacheConversation(conversationID,c);
        if (backingStore != null) {
            try {
                backingStore.writeRequest(request, conversationID, "toserver");
            } catch (IOException ioe) {
                logger.warning("Error writing server request for " + conversationID + " : " + ioe);
            }
        }
        processRequest(conversationID, c);
    }
    
    public Request getServerRequest(String conversationID) {
        Conversation c = getConversation(conversationID);
        if (c != null) {
            return c.getServerRequest();
        }
        return null;
    }
    
    public void setServerResponse(String conversationID, Response response) {
        Conversation c = (Conversation) conversation.get(conversationID);
        if (c == null) {
            c = new Conversation();
            c.setID(conversationID);
        }
        c.setServerResponse(response);
        if (backingStore != null) {
            try {
                backingStore.writeResponse(response, conversationID, "fromserver");
            } catch (IOException ioe) {
                logger.warning("Error writing server response for " + conversationID + " : " + ioe);
            }
        }
        cacheConversation(conversationID,c);
        processResponse(conversationID, c);
    }
    
    public Response getServerResponse(String conversationID) {
        Conversation c = getConversation(conversationID);
        if (c != null) {
            return c.getServerResponse();
        }
        return null;
    }
    
    public void setClientResponse(String conversationID, Response response) {
        Conversation c = (Conversation) conversation.get(conversationID);
        if (c == null) {
            c = new Conversation();
            c.setID(conversationID);
        }
        c.setClientResponse(response);
        if (backingStore != null) {
            try {
                backingStore.writeResponse(response, conversationID, "toclient");
            } catch (IOException ioe) {
                logger.warning("Error writing client response for " + conversationID + " : " + ioe);
            }
        }
        cacheConversation(conversationID,c);
    }
    
    public Response getClientResponse(String conversationID) {
        Conversation c = getConversation(conversationID);
        if (c != null) {
            return c.getClientResponse();
        }
        return null;
    }
    
    public int getConversationRow(String conversationID) {
        if (_conversationData.size() == 0) {
            return -1;
        }
        String thisID = ((String[]) _conversationData.lastElement())[_ctm.ID];
        int compare = conversationID.compareTo(thisID);
        if (compare > 0) {
            return -1 - _conversationData.size();
        } else if (compare == 0) {
            return _conversationData.size() -1;
        }
        for (int i=0; i<_conversationData.size(); i++) {
            thisID = ((String[]) _conversationData.get(i))[_ctm.ID];
            compare = conversationID.compareTo(thisID);
            if (compare < 0) {
                return -1 - i;
            } else if (compare == 0) {
                return i;
            }
        }
        logger.severe("Ran off the end of the ConversationTable looking for " + conversationID);
        return -1;
    }
    
    private void insertConversationRow(String[] rowdata) {
        int row = getConversationRow(rowdata[_ctm.ID]);
        if (row < 0) {
            _conversationData.insertElementAt(rowdata, -row -1);
            _ctm.fireTableRowsInserted(-row -1, -row -1);
        } else {
            _conversationData.set(row, rowdata);
            _ctm.fireTableRowsUpdated(row, row);
        }
    }
    
    public void setDescription(String conversationID, String description) {
        int rowIndex = getConversationRow(conversationID);
        if (rowIndex>-1) {
            String[] rowdata = (String[]) _conversationData.get(rowIndex);
            rowdata[_ctm.COMMENT]=description;
            _ctm.fireTableCellUpdated(rowIndex, _ctm.COMMENT);
        }
        if (backingStore != null) {
            try {
                backingStore.writeDescription(description, conversationID);
            } catch (IOException ioe) {
                logger.warning("Error writing description for " + conversationID + " : " + ioe);
            }
        }
    }
    
    public String getDescription(String conversationID) {
        int rowIndex = getConversationRow(conversationID);
        if (rowIndex>-1) {
            String[] rowdata = (String[]) _conversationData.get(rowIndex);
            return rowdata[_ctm.COMMENT];
        }
        return null;
    }
    
    public void setOrigin(String conversationID, String description) {
        int rowIndex = getConversationRow(conversationID);
        if (rowIndex>-1) {
            String[] rowdata = (String[]) _conversationData.get(rowIndex);
            rowdata[_ctm.ORIGIN]=description;
            _ctm.fireTableCellUpdated(rowIndex, _ctm.ORIGIN);
        }
        if (backingStore != null) {
            try {
                backingStore.writeOrigin(description, conversationID);
            } catch (IOException ioe) {
                logger.warning("Error writing origin for " + conversationID + " : " + ioe);
            }
        }
    }
    
    public String getOrigin(String conversationID) {
        int rowIndex = getConversationRow(conversationID);
        if (rowIndex>-1) {
            String[] rowdata = (String[]) _conversationData.get(rowIndex);
            return rowdata[_ctm.ORIGIN];
        }
        return null;
    }
    
    private void setChecksum(String conversationID, byte[] content) {
        MessageDigest md = null;
        try {
            md = MessageDigest.getInstance("MD5");
        } catch (NoSuchAlgorithmException nsae) {
            System.err.println("Can't calculate MD5 sums! No such algorithm!");
            System.exit(1);
        }
        String digest = TranscoderFrame.hexEncode(md.digest(content));
        int rowIndex = getConversationRow(conversationID);
        if (rowIndex>-1) {
            String[] rowdata = (String[]) _conversationData.get(rowIndex);
            rowdata[_ctm.CHECKSUM]=digest;
            _ctm.fireTableCellUpdated(rowIndex, _ctm.CHECKSUM);
        }
    }
    
    public String getChecksum(String conversationID) {
        int rowIndex = getConversationRow(conversationID);
        if (rowIndex>-1) {
            String[] rowdata = (String[]) _conversationData.get(rowIndex);
            return rowdata[_ctm.CHECKSUM];
        }
        return null;
    }
    
    private void processRequest(String conversationID, Conversation conversation) {
        Request req = conversation.getServerRequest();
        if (req == null) {
            logger.severe("Error! Conversation " + conversationID + " must have a request part!");
            return;
        }
        
        String shpp = Util.getURLSHPP(req.getURL());
        updatePath(shpp);
        URLInfo ui;
        synchronized (urlinfo) {
            ui = (URLInfo) urlinfo.get(shpp);
            synchronized (ui) {
                ui.addProperty("conversations",conversationID);
            }
            urlinfo.put(shpp,ui);
        }
        String[] rowdata = new String[_ctm.getColumnCount()];
        rowdata[_ctm.ID] = conversationID;
        rowdata[_ctm.METHOD] = req.getMethod();
        rowdata[_ctm.SHPP] = Util.getURLSHPP(req.getURL());
        rowdata[_ctm.QUERY] = Util.getURLQuery(req.getURL());
        rowdata[_ctm.COOKIE] = req.getHeader("Cookie");
        byte[] content = req.getContent();
        if (content != null) {
            rowdata[_ctm.CONTENT] = new String(content);
        }
        insertConversationRow(rowdata);
    }
    
    private void processResponse(String conversationID, Conversation conversation) {
        try {
            URL base = null;
            Request req = conversation.getServerRequest();
            base = req.getURL();
            String shpp = Util.getURLSHPP(base);
            
            URLInfo ui;
            synchronized (urlinfo) {
                ui = (URLInfo) urlinfo.get(shpp);
            }
            if (base != null) {
                Response resp = conversation.getServerResponse();
                int rowIndex = getConversationRow(conversationID);
                if (rowIndex>-1) {
                    String[] rowdata = (String[]) _conversationData.get(rowIndex);
                    rowdata[_ctm.STATUS]=resp.getStatusLine();
                    _ctm.fireTableCellUpdated(rowIndex, _ctm.STATUS);
                    rowdata[_ctm.SETCOOKIE]=resp.getHeader("Set-Cookie");
                    _ctm.fireTableCellUpdated(rowIndex, _ctm.SETCOOKIE);
                } else {
                    logger.warning("Did not find conversation " + conversationID + " in the _conversationData, got " + rowIndex);
                }
                AnalyseResponse ar = new AnalyseResponse(resp,base);
                ar.parse();
                synchronized (ui) {
                    ui.addProperty("status",resp.getStatusLine());
                    
                    Fragment[] fragments;
                    fragments = ar.getForms();
                    if (fragments != null) {
                        for (int i=0; i<fragments.length; i++) {
                            ui.addProperty("forms", conversationID + ":" + fragments[i].toString());
                        }
                    }
                    fragments = ar.getScripts();
                    if (fragments != null) {
                        for (int i=0; i<fragments.length; i++) {
                            ui.addProperty("scripts", conversationID + ":" + fragments[i].toString());
                        }
                    }
                    fragments = ar.getComments();
                    if (fragments != null) {
                        for (int i=0; i<fragments.length; i++) {
                            ui.addProperty("comments", conversationID + ":" + fragments[i].toString());
                        }
                    }
                }
                synchronized (urlinfo) {
                    urlinfo.put(shpp,ui);
                }
                URL[] links = ar.getLinks();
                if (links != null && links.length>0) {
                    URLInfo linkui;
                    URL u;
                    for (int i=0; i<links.length; i++) {
                        u = links[i];
                        if (u.getProtocol().toLowerCase().startsWith("http")) {
                            shpp = Util.getURLSHPP(u);
                            updatePath(shpp);
                            linkui = (URLInfo) urlinfo.get(shpp);
                            if (Util.getURLQuery(u) != null && linkui != null) {
                                linkui.setProperty("application",Boolean.TRUE);
                            }
                        }
                    }
                }
                links = ar.getFormAction();
                if (links != null && links.length>0) {
                    URLInfo linkui;
                    URL u;
                    for (int i=0; i<links.length; i++) {
                        u = links[i];
                        if (u.getProtocol().toLowerCase().startsWith("http")) {
                            shpp = Util.getURLSHPP(u);
                            linkui = (URLInfo) urlinfo.get(shpp);
                            if (linkui != null) {
                                linkui.setProperty("application",Boolean.TRUE);
                            }
                        }
                    }
                }
                if (resp.getStatus().equals("401")) {
                    ui.setProperty("authentication",Boolean.TRUE);
                }
                String[][] params = req.getParameters();
                if (params != null && params.length>0) {
                    String signature = req.getMethod();
                    for (int i=0; i<params.length; i++) {
                        signature = signature + ";" + params[i][0] + ":" + params[i][1];
                    }
                    ui.addProperty("parameters", signature);
                }
            }
        } catch (Exception e) {
            logger.severe("Exception in updateModel(" + conversationID + ") " + e);
        }
    }
    
    private void updatePath(String shpp) {
        URLInfo ui;
        String path = "";
        synchronized (urlinfo) {
            ui = (URLInfo)urlinfo.get(shpp);
            if (ui == null) {
                try {
                    ui = new URLInfo(shpp);
                    urlinfo.put(shpp,ui);
                } catch (MalformedURLException mue) {
                    logger.severe(mue.toString());
                    return;
                }
            }
            String[] elements = ui.getURLElements();
            for (int i=0; i<elements.length; i++) {
                path = path + elements[i];
                URLInfo pathui = (URLInfo)urlinfo.get(path);
                if (pathui == null) {
                    try {
                        pathui = new URLInfo(path);
                        urlinfo.put(path, pathui);
                        _wtm.updateNode(pathui);
                    } catch (MalformedURLException mue) {
                        logger.severe("MalformedURLException " + mue);
                        return;
                    }
                }
                synchronized (changed) {
                    if (!changed.contains(pathui)) {
                        changed.add(pathui);
                        this.setChanged();
                    }
                }
            }
            _wtm.updateNode(ui);
        }
    }
    
    private void cacheConversation(String conversationID, Conversation c) {
        if (!conversationList.contains(conversationID)) {
            if (_cachesize > 0 && conversationList.size()>=_cachesize) {
                String id = (String) conversationList.remove(0);
                conversation.remove(id);
            }
            conversationList.add(conversationID);
            conversation.put(conversationID, c);
        } // else it is already cached
    }
    
    /** Given a conversation id, returns the corresponding Conversation
     * @return the requested Conversation, or null if it does not exist
     * @param id the requested "opaque" conversation id
     */
    private Conversation getConversation(String id) {
        synchronized (conversation) {
            if (conversation.containsKey(id)) { // if we have it, return it
                conversationList.remove(id); // move it to the end
                conversationList.add(id);
                return (Conversation)conversation.get(id);
            } else if (backingStore != null) { // if we can get it, cache it
                Conversation c = backingStore.readConversation(id);
                if (c!=null) {
                    cacheConversation(id,c);
                }
                return c;
            } else {  // sorry!
                return null;
            }
        }
    }
    
    public TableModel getConversationTableModel() {
        return _ctm;
    }
    
    public TreeModel getSiteTreeModel() {
        return _wtm;
    }
    
    /** We maintain a list of URLs that we have either seen, or know of (e.g. a link
     * from a page that we HAVE seen)
     * @return how many urls we have seen, or know about
     */
    public int getNumURLs() {
        int size = 0;
        // Use an enumeration here to check whether the URL is seen or not, and return the right number
        return getURLs().length;
    }
    
    /** We maintain a list of URLs that we have either seen, or know of (e.g. a link
     * from a page that we HAVE seen)
     * @return an array of strings representing all the URLs that we know of. These URLs
     * exclude any and all parameters or fragments
     */
    public String[] getURLs() {
        ArrayList urls = new ArrayList(1);
        synchronized (urlinfo) {
            for (Iterator e = urlinfo.keySet().iterator() ; e.hasNext(); ) {
                urls.add((String) e.next());
            }
        }
        String[] urlstring = new String[urls.size()];
        for (int i=0; i< urls.size(); i++) {
            urlstring[i] = (String) urls.get(i);
        }
        return urlstring;
    }
    
    
    /** Returns an object encapsulating all we know about a particular URL
     * @param shpp the "scheme://host:port/path" making up the URL.
     * @return the URLInfo object if it exists, otherwise null
     */
    public URLInfo getURLInfo(String shpp) {
        synchronized (urlinfo) {
            if(urlinfo.containsKey(shpp)) {
                return (URLInfo) urlinfo.get(shpp);
            } else {
                logger.warning("URLInfo for " + shpp + " is null!");
                return null;
            }
        }
    }
    
    /** The BackingStore represent the necessary functions to enable us to read and
     * write what we have seen to the disk
     * @param bs The backing store object, created with an appropriate directory location.
     */
    public void setBackingStore(BackingStore bs) {
        if (bs != null) {
            this.backingStore = bs;
        }
    }
    
    public BackingStore getBackingStore() {
        return backingStore;
    }
    
    public class ConversationTableModel extends AbstractTableModel {
        
        public static final int ID = 0;
        public static final int METHOD = 1;
        public static final int SHPP = 2;
        public static final int QUERY = 3;
        public static final int COOKIE = 4;
        public static final int CONTENT = 5;
        public static final int STATUS = 6;
        public static final int SETCOOKIE = 7;
        public static final int CHECKSUM = 8;
        public static final int COMMENT = 9;
        public static final int ORIGIN = 10;
        
        protected String [] columnNames = {
            "ID", "Method", "Path", "Query",
            "Cookies", "Content", "Status",
            "Set-Cookie", "Checksum", "Comment",
            "Origin"
        };
        
        private Logger logger = Logger.getLogger("za.org.dragon.exodus.ConversationTableModel");
        
        public ConversationTableModel() {
        }
        
        public String getColumnName(int column) {
            if (column < columnNames.length) {
                return columnNames[column];
            }
            return "";
        }
        
        public synchronized int getColumnCount() {
            return columnNames.length;
        }
        
        public synchronized int getRowCount() {
            return _conversationData.size();
        }
        
        public synchronized Object getValueAt(int row, int column) {
            if (row<0 || row >= _conversationData.size()) {
                throw new ArrayIndexOutOfBoundsException("Attempt to get row " + row + ", column " + column + " : row does not exist!");
            }
            String[] rowdata = (String[]) _conversationData.get(row);
            if (column <= columnNames.length) {
                return rowdata[column];
            } else {
                throw new ArrayIndexOutOfBoundsException("Attempt to get row " + row + ", column " + column + " : column does not exist!");
            }
        }
    }
    
    private class WebTreeModel extends DefaultTreeModel {
        
        private DefaultTreeModel treeModel;
        private DefaultMutableTreeNode root;
        private Map treeNodes = Collections.synchronizedMap(new TreeMap());
        
        /** Creates a new instance of WebTreeModel */
        public WebTreeModel() {
            super(null, true);
            root = new DefaultMutableTreeNode(null);
            super.setRoot(root);
            root.setAllowsChildren(true);
            treeNodes.put("",root);
        }
        
        public void clear() {
            root.removeAllChildren();
        }
        
        protected TreePath updateNode(URLInfo ui) {
            String shpp = ui.getURL().toString();
            DefaultMutableTreeNode un;
            synchronized (treeNodes) {
                un = (DefaultMutableTreeNode)treeNodes.get(shpp);
                if (un != null) {
                    un.setUserObject(ui);
                    treeNodes.put(shpp,un);
                    fireTreeNodesChanged(un, un.getPath(), null, null);
                    return new TreePath(un.getPath());
                }
            }
            String[] elements = ui.getURLElements();
            String path = "";
            synchronized (treeNodes) {
                DefaultMutableTreeNode parent = root;
                for (int i = 0; i<elements.length-1; i++) {
                    path = path + elements[i];
                    parent = (DefaultMutableTreeNode)treeNodes.get(path);
                    if (parent == null) {
                        logger.severe("ERROR: an intermediate node was null! path is \"" + path + "\"");
                        System.exit(0);
                    }
                }
                path = path + elements[elements.length-1];
                un = new DefaultMutableTreeNode(ui);
                if (path.endsWith("/")) {
                    un.setAllowsChildren(true);
                } else {
                    un.setAllowsChildren(false);
                }
                treeNodes.put(path,un);
                
                int numChildren = parent.getChildCount();
                if (numChildren == 0) { // this is the first child, just add it
                    parent.add(un);
                    fireTreeNodesInserted(parent, parent.getPath(), new int[] {numChildren}, new Object[] {un});
                } else { // work out where to put it
                    DefaultMutableTreeNode node = (DefaultMutableTreeNode)parent.getLastChild();
                    URLInfo urlinfo = (URLInfo)node.getUserObject();
                    String siblingPath = urlinfo.getURL().toString();
                    if (path.compareTo(siblingPath) > 0 ) { // If it is greater than the last node, add it and be done
                        parent.add(un);
                        fireTreeNodesInserted(parent, parent.getPath(), new int[] {numChildren}, new Object[] {un});
                    } else { // work out where to insert it
                        for (int i = 0; i<numChildren; i++) {
                            node = (DefaultMutableTreeNode)parent.getChildAt(i);
                            urlinfo = (URLInfo)node.getUserObject();
                            siblingPath = urlinfo.getURL().toString();
                            int c = path.compareTo(siblingPath);
                            if (c < 0) {
                                parent.insert(un,i);
                                fireTreeNodesInserted(parent, parent.getPath(), new int[] {i}, new Object[] {un});
                                break;
                            } else if ( c == 0) {
                                break;
                            }
                        }
                    }
                }
            }
            if (un != null) {
                return new TreePath(un.getPath());
            } else {
                return new TreePath(root);
            }
        }
        
    }
    
}
