#!/usr/bin/perl -w
# mangle.pl 0.5
# Written by Rogan Dawes <rdawes@deloitte.co.za>
#
# Based on code by Lluis Mora, in httpush 0.9b8 - heavily rewritten
# # HTTPush v0.9b8 by Lluis Mora <llmora@s21sec.com>
#
# Very little (no) error checking is done for command line options :-(
use Data::Dumper;                      # for debugging

use IO::Socket;                        # For the listen/accept
use LWP;                               # to fetch the pages
$DO_SSL=1;                             # Whether we support SSL or not
if ($DO_SSL) {
  print STDERR "Using SSL modules\n";
  use Crypt::SSLeay;                     # to add https support to LWP
  use Net::SSLeay qw(die_now die_if_ssl_error); # to fake the SSL server
}

use Digest::MD5 qw(md5_hex);           # used to identify changes in responses
use Getopt::Std;                       # for command line options
my %opt; # to hold the options

$|=1;                                  # set unbuffered

my ($ua, $port, $listen_port);
my $debugline=0;
my $logline=0;

$SIG{INT} = \&getout;
$SIG{CHLD}='IGNORE';

# Create the user agent object
$ua = LWP::UserAgent->new;
$ua->env_proxy(); # pick up environment proxy settings (man LWP::UserAgent)

getopts("df:hiI:m:p:",\%opt);

if($opt{'h'}) {
  &usage();
  exit;
}

if($opt{'p'}) { $listen_port=$opt{'p'}; } 
else { $listen_port=8000; }

if($opt{'I'}) { $listen_on=$opt{'I'}; }
else { $listen_on="0.0.0.0"; } # all interfaces (not good for security :-)

if($opt{'f'}) {
  if ($opt{'f'} eq '-') {
    open(LOGFILE, ">&STDOUT") || die "Can't dup STDOUT! $!\n";
  } else {
    if ( ! -e $opt{'f'} ) {
      mkdir $opt{'f'} || die "Can't create log directory $opt{'f'}\n";
    }
    if ( -d $opt{'f'} ) {
      $opt{'f'}.='/log';
    }
    open (LOGFILE,">>$opt{'f'}") ||
      die "Can't open logfile $opt{'f'}\n";
  }
  autoflush LOGFILE;
} else { print STDERR "=> Not logging any traffic\n"; }

if ($DO_SSL) { initssl(); }

my $Server = IO::Socket::INET->new(
  Listen    => 10, 
  LocalAddr => $listen_on,
  LocalPort => $listen_port,
  Reuse     => 1,
  Proto     => 'tcp') || die "Can't create server socket.";

print STDERR "=> Listening on $listen_on:$listen_port\n";

# Enter listen/accept loop
while (1) {
  my $paddr = accept(Client,$Server);
  if (defined $paddr) {
    ($port,$iaddr) = sockaddr_in($paddr);
    $addr=inet_ntoa($iaddr);
    Debug("Got a connection from $addr:$port");
    spawn();
    close Client;
  }
}

print STDERR "Finished the accept loop somehow!";
if ($DO_SSL) { Net::SSLeay::CTX_free ($ctx); }
close $Server;
close Client;
exit(0);

############################# Functions ######################################
sub initssl { # Initialize SSL

  Net::SSLeay::load_error_strings();
  Net::SSLeay::SSLeay_add_ssl_algorithms();
  Net::SSLeay::randomize(0);

  $ctx = Net::SSLeay::CTX_new ()         or die_now("CTX_new ($ctx): $!");

  Net::SSLeay::CTX_set_options($ctx, &Net::SSLeay::OP_ALL)
      and die_if_ssl_error("ssl ctx set options");

  # Following will ask password if private key is encrypted
  Net::SSLeay::CTX_use_RSAPrivateKey_file ($ctx, 'mangle.crt', &Net::SSLeay::FILETYPE_PEM);
  die_if_ssl_error("private key");

  Net::SSLeay::CTX_use_certificate_file ($ctx, 'mangle.crt', &Net::SSLeay::FILETYPE_PEM);
  die_if_ssl_error("certificate");
}

sub spawn { # does the fork
  my $pid;

  if(!defined($pid = fork)) {
    Debug("Fork failed!");
    return;
  } elsif($pid) {
    return;            # I'm the parent
  }
  
  $debugline=0;
  $logline=0;

  if ($opt{'m'}) {
  # Look for an include file, otherwise we are just a logging proxy.
  # We put this after the fork so that if the include file is modified, we don't
  # have to restart the proxy
  # If it fails, just proxy the request, don't die.
    if (! (do $opt{'m'})) {
      Log("couldn't parse $opt{'m'}: $@") if $@;
    }
  }

  handleconnection();
  Log("Done");
  exit;
}

sub handleconnection { # does the clientrequest, mangle, serverrequest, mangle, clientresponse cycle

  $request=handlerequest();
  if ($request->method() eq "CONNECT" && !$DO_SSL) {
    Debug("Can't handle an SSL connection!");
    $response=new HTTP::Response(405);
    $response->request($request);
    $response->push_header("Allow","GET, HEAD, PUT, POST");
  } else {
    if (defined &manglerequest) { $request=manglerequest($request); }
  
    $response=handleresponse($request);
  
    if (defined &mangleresponse) { $response=mangleresponse($response); }
  }

  sendresponse($response);
  
  # if we were to support "Connection: keep-alive", we should make this a 
  # loop with a timeout, rather than exiting here. We should also check for
  # recursive CONNECT methods.
  exit(0);
}

sub getrequest { # Returns a Perl HTTP::Request object containing the client's request
  my $host=shift;
  my $https=0;
  if (defined $host) { $https=1; }

  my ($bytes,$in,$read,$req,$method,$uri,$proto,$content);
  my $request = new HTTP::Request;

  $in='';
  $end=0;
  while (!$end && (($bytes,$got)=client_read($https)) && $bytes ) {
    $in.=$got;

    my $i=index($in,"\n");
    while ($i>-1 && !$end) { # We still have a line to read
      $line=substr($in,0,$i+1,''); # extract the next line
      $line=~s/\n//g; # get rid of CR
      $line=~s/\r//g; # get rid of LF
#      Debug("Request: '$line'");
      if (!defined $method) { # This is the first line
        # Split the request into method, URI and proto
        ($method, $uri, $proto)=split(/\s+/,$line); 
        $request->method($method);
        $request->uri($uri);
        $request->protocol($proto);

        # $host is passed in if we have faked an SSL session
        if ($https) { 
          # Make SSL URLs absolute
          $request->uri($request->uri->abs("https://".$host)); 
        }
      } elsif ($line=~/([^:]+)\s*:\s*(.*)/) { # This is a header line
        my $header=lc($1);
        my $value=$2;
        # Not sure if this is important
        if (($header eq "connection") || ($header eq "proxy-connection")) {
#          Debug("Ignoring client header $header : $value");
#        } elsif (($header eq "if-modified-since") || ($header eq "if-none-match")) {
#          Debug("Ignoring client header $header : $value");
        } else {
          $request->push_header($header,$value);
        }
      } elsif ($line eq '') { # We have the blank line that signals the end of the headers
#        Debug("Got a blank line");
        $end=1;
#        Debug("Content Length: ".$request->content_length);
        if (defined $request->content_length) {
          # Initialise $content in case we read too much doing the headers
          $content=substr($in,0,$request->content_length,''); 
          while ((length($content)<$request->content_length) && 
                 (($bytes,$got)=client_read($https)) && $bytes) {
            $content.=$got;
          }
          if (length($content)<$request->content_length) {
            Debug("Insufficient content in ".$request->as_string("\r\n"));
            exit(1);
          } else {
            $request->content($content);
          }
        }
      } else { # This is something I don't understand
        Debug("Exiting with strange header content: $line");
        exit(1);
      }
      $i=index($in,"\n");
    }
  }
  if (!$end) {
    $request->method('');
  }
  return $request;
}

sub handlerequest { # obtains a request object (reflecting the client request) and logs it 
  my $request=getrequest();

  # Asking for an SSL proxy connection ?
  if ($DO_SSL && ($request->method() eq "CONNECT")) { 
    Debug("Faking an SSL session");
    $ssl = Net::SSLeay::new($ctx)      or die_now("SSL_new ($ssl): $!");
    Net::SSLeay::set_fd($ssl, fileno(Client));
    Net::SSLeay::print_errs();

    $https=0; # Not yet set, write the response first
    client_write($https,"HTTP/1.1 200 OK\r\n\r\n") || die "Error writing to client socket\n";
    my $err = Net::SSLeay::accept($ssl) and die_if_ssl_error('ssl accept');

    Debug("Fake SSL session established (".Net::SSLeay::get_cipher($ssl).")");

    $https=1;
    $request=getrequest($request->uri->as_string);
  }
 
  if (!defined $request->uri) { # No data sent
    Debug("They just wanted our cert");
    exit;
  }

  if ($request->method() eq "") { # No data sent
    Debug("Didn't complete the request");
    exit;
  }
 
  my $uri=$request->uri->as_string();
  my $path=$request->uri->path();
  if (!$opt{'i'} && $path=~/\.(gif|jpg|png)$/i) {
#    Debug("Not Recording images");
    undef $opt{'d'};
    undef $opt{'f'};
  }
  Record("fromclient",$request);
  Log($request->method()." ".$request->uri->as_string());
  
  if (defined $request->header("cookie")) {
    Log("Cookie : ".$request->header("cookie"));
  }

  if (defined $request->header("proxy-authenticate")) {
    Log("Proxy-Authenticate : ".$request->header("cookie"));
  }

  if (defined $request->header("www-authenticate")) {
    Log("WWW-Authenticate : ".$request->header("cookie"));
  }

  if ($request->method() eq "POST") {
    Log("Content: ".$request->content());
   }

  return $request;
}

sub handleresponse { # Fetch response from server
  my $request=shift;
  
# Fetch request from server
# must use simple_request, as the Perl LWP method of following redirects
# is different to IE and Netscape/Mozilla
  my $response=$ua->simple_request($request);

# We should really strip out the extra headers that LWP adds
# Too much effort at the moment, and the browser does nothing with them
# They can be useful, especially for recording SSL negotiation options
  Record("fromserver",$response);

  # we record a checksum of the content to assist the analysis process
  # we can skip message bodies that have the same MD5
  # we do this before the status is logged, because analysis will be
  # triggered on seeing the status line
  Log("MD5 ".md5_hex($response->content));

  my $status=$response->status_line;
  # LWP adds a CR to the error message. Makes the log untidy :-(
  $status=~s/\n//g; $status=~s/\r//g;
  Log($status);
  if (defined $response->header("set-cookie")) {
    Log("Set-Cookie: ".$response->header("set-cookie"));
  }

# IE has a haffy fit if you offer NTLM auth through a proxy
  if (defined $response->proxy_authenticate && 
      $response->proxy_authenticate eq "NTLM") {
    $response->header("proxy-authenticate" => "Basic");
  }

  return $response;
}

sub sendresponse { # send the response back to the client
  my $response=shift;
  #  if ($response->request->as_string("\r\n")=~/https/) { $https=1; }
  client_write($https,$response->as_string("\r\n"));
}

sub Log { # Prints parameters to logfile and STDERR if logging and debugging are enabled
  my ($data)=@_;

  if ($opt{'d'}) {
    printf STDERR "%5d %3d ! %s\n",$$,$debugline++,$data;
  }
  if ($opt{'f'}) {
    printf LOGFILE "%5d %3d # %s\n",$$,$logline++,$data;
  }
}

sub Debug { # Prints parameters to STDERR if debugging is enabled
  my ($data)=@_;

  if ($opt{'d'}) {
    printf STDERR "%5d %3d ! %s\n",$$,$debugline++,$data;
  }
}

sub Record { # Logs data to an audit file
  my $label=shift;
  my $object=shift; # expects an HTTP::Request or HTTP::Response object

  if ($opt{'f'} && $opt{'f'} ne '-') {
    open (CONTENT,">>$opt{'f'}-$$-$label") || 
      die "Couldn't log content : $!\n";
    print CONTENT $object->as_string("\r\n");
    close CONTENT;
  }
}

sub getout { # Handles SIGINT "Ctrl-C"
  if($opt{'f'}) {
    close(LOGFILE);
  }
  exit 1;
}

sub client_read { # Reads from socket or SSL, depending
  my $https=shift;
   my ($bytes, $got)=(0,'');

  if (!$https) {
    $bytes=sysread(Client,$got,1024);
  } else {
    $got=Net::SSLeay::read($ssl);
    if (defined $got) { $bytes=length($got); } else { $bytes=0; }
  }
  return ($bytes,$got);
}

sub client_write { # Writes to socket or SSL, depending
  my $https=shift;
  my $string=shift;
  my $bytes=0;
  my $wrote;
  my $total=length($string);

  if (!$https) {
#    Debug("client_write: ".$string);
    while ($bytes < $total) {
      $wrote=syswrite(Client,$string,length($string));
      if ($wrote) {
        substr($string,0,$wrote,'');
        $bytes+=$wrote;
      } else {
        return 0;
      }
    }
  } else {
#    Debug("client_write(ssl): ".$string);
    Net::SSLeay::ssl_write_all($ssl, $string) or die "ssl write failure";
  }
  return 1;
}

sub usage() { # Standard usage blurb
  print STDERR "Mangle - a logging HTTP and HTTPS proxy for web application analysis\n\n";
  print STDERR 'Usage: mangle [-d] [-f logfile] [-h] [-I address] [-m includefile] [-p port] 
 -d         Debug mode. Show debug information.
 -f file    Use log file "file". This text is also the file prefix for the content logged. 
            (default no logging)
 -h         Display usage.
 -i         log images i.e. requests with an extension of .gif|jpg|png (off by default)
 -I address Listen on local IP address (default 0.0.0.0)
 -m file    source "manglerequest" and "mangleresponse" functions from "file"
            "file" must return a non-zero value when sourced. i.e. last line has "1;"
            functions sourced can use any functions in the main program e.g. Record, etc
 -p port    Listen on port "port" (default 8000)
';

  exit(1);
}
